package nemosofts.voxradio.activity;

import android.annotation.SuppressLint;
import android.content.Intent;
import android.content.res.ColorStateList;
import android.graphics.Bitmap;
import android.graphics.drawable.Drawable;
import android.os.Bundle;
import android.util.Log;
import android.view.View;
import android.view.Window;
import android.view.WindowManager;
import android.widget.ImageView;
import android.widget.ProgressBar;
import android.widget.TextView;
import android.widget.Toast;

import androidx.activity.EdgeToEdge;
import androidx.activity.OnBackPressedCallback;
import androidx.annotation.NonNull;
import androidx.annotation.OptIn;
import androidx.core.content.ContextCompat;
import androidx.core.graphics.Insets;
import androidx.core.view.ViewCompat;
import androidx.core.view.WindowCompat;
import androidx.core.view.WindowInsetsCompat;
import androidx.core.view.WindowInsetsControllerCompat;
import androidx.media3.common.util.UnstableApi;
import androidx.nemosofts.AppCompatActivity;
import androidx.nemosofts.material.ImageViewRound;
import androidx.nemosofts.material.SnowfallView;
import androidx.nemosofts.theme.ColorUtils;
import androidx.nemosofts.material.BlurImage;
import androidx.nemosofts.utils.NetworkUtils;
import androidx.palette.graphics.Palette;

import com.nemosofts.swipebutton.SwipeButton;
import com.squareup.picasso.Picasso;
import com.squareup.picasso.Target;

import org.greenrobot.eventbus.Subscribe;
import org.greenrobot.eventbus.ThreadMode;

import nemosofts.voxradio.R;
import nemosofts.voxradio.callback.Callback;
import nemosofts.voxradio.callback.Method;
import nemosofts.voxradio.executor.LoadStatus;
import nemosofts.voxradio.interfaces.SuccessListener;
import nemosofts.voxradio.item.ItemRadio;
import nemosofts.voxradio.utils.GlobalBus;
import nemosofts.voxradio.utils.IfSupported;
import nemosofts.voxradio.utils.MessageEvent;
import nemosofts.voxradio.utils.helper.Helper;
import nemosofts.voxradio.utils.helper.SPHelper;

public class DriveModeActivity extends AppCompatActivity implements View.OnClickListener {

    private static final String TAG = "DriveModeActivity";
    private Helper helper;
    private SPHelper spHelper;
    private TextView title;
    private ImageView blurBg;
    private ImageView previous;
    private ImageView next;
    private ImageView play;
    private ImageView fav;
    private ProgressBar pbLoading;
    private ImageViewRound poster;

    @Override
    public void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        IfSupported.isRTL(this);
        IfSupported.isScreenshot(this);
        hideNavigationBarStatusBars();
        EdgeToEdge.enable(this);
        ViewCompat.setOnApplyWindowInsetsListener(findViewById(R.id.bg_page), (v, insets) -> {
            Insets systemBars = insets.getInsets(WindowInsetsCompat.Type.systemBars());
            v.setPadding(systemBars.left, systemBars.top, systemBars.right, systemBars.bottom);
            return insets;
        });

        helper = new Helper(this);
        spHelper = new SPHelper(this);

        if (Boolean.TRUE.equals(spHelper.isDriveKeepScreen())){
            getWindow().addFlags(WindowManager.LayoutParams.FLAG_KEEP_SCREEN_ON);
        }

        title = findViewById(R.id.tv_title_drive_mode);
        blurBg = findViewById(R.id.iv_drive_mode);

        pbLoading = findViewById(R.id.pb_music_loading);
        play = findViewById(R.id.iv_music_play);
        previous = findViewById(R.id.iv_music_previous);
        next = findViewById(R.id.iv_music_next);
        fav = findViewById(R.id.iv_drive_mode_fav);
        poster = findViewById(R.id.iv_drive_mode_add);

        play.setOnClickListener(this);
        previous.setOnClickListener(this);
        next.setOnClickListener(this);
        fav.setOnClickListener(this);

        findViewById(R.id.iv_drive_mode_close).setOnClickListener(view -> finish());

        SwipeButton enableButton = findViewById(R.id.swipe_btn);
        enableButton.setOnStateChangeListener(active -> {
            if (active){
                finish();
            }
        });

        SnowfallView snowFall = findViewById(R.id.drive_snow_fall);
        if (Boolean.TRUE.equals(spHelper.isDriveSnowFall())){
            snowFall.restartFalling();
        } else {
            snowFall.stopFalling();
        }
        snowFall.setVisibility(Boolean.TRUE.equals(spHelper.isDriveSnowFall()) ? View.VISIBLE : View.GONE);

        OnBackPressedCallback callback = new OnBackPressedCallback(true) {
            @Override
            public void handleOnBackPressed() {
                // this method is empty
            }
        };
        getOnBackPressedDispatcher().addCallback(this, callback);
    }

    @Override
    public int setContentViewID() {
        return R.layout.activity_drive_mode;
    }

    @SuppressLint("SetTextI18n")
    public void changeText(@NonNull ItemRadio itemRadio) {
        title.setText(itemRadio.getRadioTitle());
        changeFav(itemRadio.getIsFav());
        try {
            Target target = new Target() {
                @Override
                public void onBitmapLoaded(Bitmap bitmap, Picasso.LoadedFrom from) {
                    try {
                        int blurAmount = spHelper.getBlurAmountDrive();
                        blurBg.setImageBitmap(BlurImage.fastBlur(bitmap, 1f, blurAmount));
                    } catch (Exception e) {
                        Log.e(TAG, "Error in onBitmapLoaded: ", e);
                    }
                }
                @Override
                public void onBitmapFailed(Exception e, Drawable errorDrawable) {
                    blurBg.setImageResource(R.drawable.material_design_default);
                }
                @Override
                public void onPrepareLoad(Drawable placeHolderDrawable) {
                    // this method is empty
                }
            };
            blurBg.setTag(target);
            Picasso.get()
                    .load(itemRadio.getImage())
                    .placeholder(R.drawable.material_design_default)
                    .error(R.drawable.material_design_default)
                    .into(target);
        } catch (Exception e) {
            Log.e(TAG, "Error in onBitmapLoaded: ", e);
        }

        Picasso.get()
                .load(itemRadio.getImage())
                .placeholder(R.drawable.material_design_default)
                .error(R.drawable.material_design_default)
                .into(poster);

        if (Boolean.TRUE.equals(spHelper.isDriveColor())){
            setColorText(itemRadio);
        }
    }

    private void setColorText(ItemRadio itemRadio) {
        try {
            Picasso.get()
                    .load(itemRadio.getImage())
                    .centerCrop()
                    .resize(100, 100)
                    .into(new Target() {
                        @Override
                        public void onBitmapLoaded(Bitmap bitmap, Picasso.LoadedFrom from) {
                            Palette.from(bitmap).generate(palette -> {
                                if (palette != null){
                                    Palette.Swatch textSwatch = palette.getVibrantSwatch();
                                    if (textSwatch == null) {
                                        return;
                                    }
                                    try {
                                        View view1 = findViewById(R.id.view_1);
                                        View view2 = findViewById(R.id.view_2);
                                        view1.setBackgroundTintList(ColorStateList.valueOf(textSwatch.getRgb()));
                                        view2.setBackgroundTintList(ColorStateList.valueOf(textSwatch.getRgb()));
                                    } catch (Exception e) {
                                        Log.e(TAG, "Error in setColorText Palette ", e);
                                    }
                                }
                            });
                        }
                        @Override
                        public void onBitmapFailed(Exception e, Drawable errorDrawable) {
                            // this method is empty
                        }
                        @Override
                        public void onPrepareLoad(Drawable placeHolderDrawable) {
                            // this method is empty
                        }
                    });
        } catch (Exception e) {
            Log.e(TAG, "Error in setColorText", e);
        }
    }

    public void changeFav(Boolean isFav) {
        fav.setImageDrawable(ContextCompat.getDrawable(this, Boolean.TRUE.equals(isFav)
                ? R.drawable.ic_heart_fill
                : R.drawable.ic_heart_line));
        fav.setColorFilter(ColorUtils.colorWhite(this));
    }

    @Override
    public void onClick(View view) {
        if (view == null){
            return;
        }
        int id = view.getId();
        if (id == R.id.iv_music_play){
            playPause();
        } else if (id == R.id.iv_music_previous){
            previous();
        } else if (id == R.id.iv_music_next){
            next();
        } else if (id == R.id.iv_drive_mode_fav){
            doFav();
        }
    }

    private void doFav() {
        if (!spHelper.isLogged()) {
            helper.clickLogin();
            return;
        }

        if (Callback.getArrayListPlay().isEmpty()) {
            showSnackBar(getString(R.string.error_no_radio_selected));
            return;
        }

        if (!NetworkUtils.isConnected(this)) {
            showSnackBar(getString(R.string.err_internet_not_connected));
            return;
        }

        LoadStatus loadFav = new LoadStatus(new SuccessListener() {
            @Override
            public void onStart() {
                changeFav(!Callback.getArrayListPlay().get(Callback.getPlayPos()).getIsFav());
            }

            @Override
            public void onEnd(String success, String favSuccess, String message) {
                if (success.equals("1")) {
                    Callback.getArrayListPlay().get(Callback.getPlayPos()).setIsFav(message.equals("Added to Favourite"));
                    changeFav(Callback.getArrayListPlay().get(Callback.getPlayPos()).getIsFav());
                    showSnackBar(message);
                } else {
                    showSnackBar(getString(R.string.err_server_not_connected));
                }
            }
        }, helper.getAPIRequest(Method.METHOD_DO_FAV, 0,
                Callback.getArrayListPlay().get(Callback.getPlayPos()).getId(),
                "", "", "", spHelper.getUserId(), "",
                "", "", "", "", "", "", null));
        loadFav.execute();
    }

    private void showSnackBar(@NonNull String message) {
        Toast.makeText(DriveModeActivity.this, message, Toast.LENGTH_SHORT).show();
    }

    @OptIn(markerClass = UnstableApi.class)
    public void playPause() {
        if (Boolean.TRUE.equals(Callback.getIsPlayed())) {
            handlePlayerAction(PlayerService.ACTION_TOGGLE);
        } else {
            handlePlayerAction(PlayerService.ACTION_PLAY);
        }
    }

    @OptIn(markerClass = UnstableApi.class)
    public void previous() {
        handlePlayerAction(PlayerService.ACTION_PREVIOUS);
    }

    @OptIn(markerClass = UnstableApi.class)
    public void next() {
        handlePlayerAction(PlayerService.ACTION_NEXT);
    }

    @OptIn(markerClass = UnstableApi.class)
    private void handlePlayerAction(String action) {
        if (!Callback.getArrayListPlay().isEmpty()) {
            if (NetworkUtils.isConnected(this)) {
                Intent intent = new Intent(DriveModeActivity.this, PlayerService.class);
                intent.setAction(action);
                startService(intent);
            } else {
                showSnackBar(getString(R.string.err_internet_not_connected));
            }
        } else {
            showSnackBar(getString(R.string.error_no_radio_selected));
        }
    }

    public void changePlayPauseIcon(Boolean isPlay) {
        if (Boolean.FALSE.equals(isPlay)) {
            play.setImageDrawable(ContextCompat.getDrawable(this, R.drawable.ic_play));
        } else {
           play.setImageDrawable(ContextCompat.getDrawable(this, R.drawable.ic_pause));
        }
    }

    public void isBuffering(boolean isBuffer) {
        if (isBuffer) {
            play.setVisibility(View.GONE);
            pbLoading.setVisibility(View.VISIBLE);
        } else {
            play.setVisibility(View.VISIBLE);
            pbLoading.setVisibility(View.GONE);
            changePlayPauseIcon(true);
        }
        play.setEnabled(!isBuffer);
        next.setEnabled(!isBuffer);
        previous.setEnabled(!isBuffer);
    }

    @Subscribe(sticky = true, threadMode = ThreadMode.MAIN)
    public void onSongChange(ItemRadio itemRadio) {
        changeText(itemRadio);
    }

    @Subscribe(sticky = true, threadMode = ThreadMode.MAIN)
    public void onBufferChange(@NonNull MessageEvent messageEvent) {
        if (messageEvent.message.equals("buffer")) {
            isBuffering(messageEvent.flag);
        } else {
            changePlayPauseIcon(messageEvent.flag);
        }
    }

    @Override
    public void onStart() {
        super.onStart();
        GlobalBus.getBus().register(this);
    }

    @Override
    public void onStop() {
        GlobalBus.getBus().unregister(this);
        super.onStop();
    }

    public void hideNavigationBarStatusBars() {
        try {
            Window window = getWindow();
            View decorView = window.getDecorView();

            // Allow layout to extend behind system bars
            WindowCompat.setDecorFitsSystemWindows(window, false);

            // Use AndroidX controller to manage insets across all versions
            WindowInsetsControllerCompat controller =
                    new WindowInsetsControllerCompat(window, decorView);

            // Hide both status and navigation bars
            controller.hide(WindowInsetsCompat.Type.statusBars() | WindowInsetsCompat.Type.navigationBars());

            // Enable immersive sticky (swipe to show system bars temporarily)
            controller.setSystemBarsBehavior(
                    WindowInsetsControllerCompat.BEHAVIOR_SHOW_TRANSIENT_BARS_BY_SWIPE
            );
        } catch (Exception e) {
            Log.e("DriveMode", "Failed to hide Navigation Bar & Status Bar", e);
        }
    }
}